/* BSD license throughout
 * Ville Timonen, 2015
 * wili@wili.cc
 */
#include <cstdio>
#include <cstdlib>
#include <unistd.h>
#include <string>
#include <cmath>
#include "point.h"
#include "imgwriter.h"

// Comment out to not export bias.  Otherwise it's outputted as the .z component, value is the range
#define EXPORT_BIAS 2.0

// Comment out for no PNG output (instead print on screen)
#define EXPORT_FILE "16x9.png"

// Interleaved sets
#define SETS 9 

// Points in each set
#define POINTS 16

// Uber cool ascii art
//#define VISUALIZE_ASCII

#define SAMPLE_SIZE 256 // You don't want to touch this


// Rewrite this to have the layout you want
#ifdef EXPORT_FILE
void writeImage( std::vector< PointSet > psSets )
{
	int iWidth = int( sqrtf( (float)SETS ) + 0.5f );
	int iHeight = SETS * POINTS / iWidth;

	int *pXY = (int*)malloc( sizeof(int) * iWidth * iHeight * 2 );
	float *pBias = (float*)malloc( sizeof(float) * iWidth * iHeight );

	for( int y = 0; y < iHeight; ++y )
	{
		for( int x = 0; x < iWidth; ++x )
		{
			int iIndex = y * iWidth + x;

			int iSetY = y % iWidth;
			int iSet = iSetY * iWidth + x;

			int iPoint = y / iWidth;

			Point p = psSets.at( iSet ).getPoint( iPoint );

			pXY[ iIndex * 2 + 0 ] = p.getX();
			pXY[ iIndex * 2 + 1 ] = p.getY();

			#ifdef EXPORT_BIAS
			pBias[ iIndex ] = p.getWeight() / EXPORT_BIAS;
			#endif
		}
	}

	writeImage( iWidth, iHeight, pXY,
			#ifdef EXPORT_BIAS
			pBias,
			#else
			NULL,
			#endif
			std::string( EXPORT_FILE ) );

	free( pXY );
	free( pBias );
}
#endif

// Density as a function of distance from the origo.
// Has to return values for r e [0, 1], but the result does not have to be normalized
float getDensity( float r )
{
	return 1.0f / ( 1.0f + r * r );
}

int g_colorTable[SAMPLE_SIZE][SAMPLE_SIZE];
int g_sampleTable[SAMPLE_SIZE][SAMPLE_SIZE];
const float g_fRadius = 0.5f;
const int g_iGatherY = 2;
const char *g_infoLine;

int getColorIndex( int iIndex )
{
	int iMod = iIndex % 5;
	if( iMod > 1 )
		iMod++;

	return iMod + 31;
}

void clearSampleTable()
{
	for( int y = 0; y < SAMPLE_SIZE; ++y )
		for( int x = 0; x < SAMPLE_SIZE; ++x )
		{
			Point p( x, y );
			bool bIsWithinRadius = p.isWithinRadius();

			g_sampleTable[x][y] = bIsWithinRadius ? ' ' : 'X';
			g_colorTable[x][y] = bIsWithinRadius ? 0 : 0;
		}
}

void draw()
{
	printf( "\e[1,1mSample space %dx%d, %d sets, %d points each -- ", SAMPLE_SIZE, SAMPLE_SIZE, SETS, POINTS );
	printf( g_infoLine );
	printf( "\e[1,0m\n" );

	for( int x = 0; x < SAMPLE_SIZE + 2; ++x )
		printf( "-" );
	printf( "\n" );
	for( int y = 0; y < SAMPLE_SIZE; y += g_iGatherY )
	{
		int sampleLine[SAMPLE_SIZE];
		int colorLine[SAMPLE_SIZE];

		printf( "|" );
		for( int x = 0; x < SAMPLE_SIZE; ++x )
		{
			sampleLine[x] = ' ';
			colorLine[x] = 0;
		}

		for( int yg = y; yg < y + g_iGatherY; ++yg )
			for( int x = 0; x < SAMPLE_SIZE; ++x )
				if( g_sampleTable[x][yg] != ' ' )
				{
					sampleLine[x] = g_sampleTable[x][yg];
					colorLine[x] = g_colorTable[x][yg];
				}

		for( int x = 0; x < SAMPLE_SIZE; ++x )
		{
			printf( "\e[1;%dm%c", colorLine[x], sampleLine[x] );
		}

		printf( "|" );
		printf( "\n" );
	}
	for( int x = 0; x < SAMPLE_SIZE + 2; ++x )
		printf( "-" );
	printf( "\n" );
}

int main()
{
	Point::setCenter( g_fRadius );
	Point::setSampleSize( SAMPLE_SIZE );
	PointSet::setSampleSize( SAMPLE_SIZE );

	clearSampleTable();

	std::vector< PointSet > psSets;
	PointSet ps;

	for( int p = 0; p < SETS * POINTS; ++p )
	{
		ps.addPoint();
		#ifdef VISUALIZE_ASCII
		char infoLine[1024];
		sprintf( infoLine, "GENERATING SAMPLES (%d/%d)", p + 1, SETS * POINTS );
		g_infoLine = infoLine;
		clearSampleTable();
		//psSets.at( i ).fillPoints( ( i % 6 ) + 31, SAMPLE_SIZE, &g_colorTable[0][0], &g_sampleTable[0][0] );
		ps.fillPoints( getColorIndex( 0 ), SAMPLE_SIZE, &g_colorTable[0][0], &g_sampleTable[0][0] );
		draw();
		#endif
	}

	// Each set filled in sequence (the last set takes the fall)
	#if 0
	for( int s = 0; s < SETS; ++s )
	{
		PointSet thisSet = ps.subSet( points );
		psSets.push_back( thisSet );

		ps.removePoints( thisSet );
	}
	#endif

	// Round robin picking (distribute badness across all sets)
	#if 1
	for( int s = 0; s < SETS; ++s )
		psSets.push_back( PointSet() );

	int iNextSet = 0;
	while( ps.getPointCount() )
	{
		ps.transferPointTo( &psSets[ iNextSet ] );

		iNextSet = ( iNextSet + 1 ) % SETS;
	}
	#endif

	// Sorting sets spatially (same indices across the sets map to closest locations)
	{
		std::vector< PointSet > sortedPoints;

		// The first set is unchanged
		sortedPoints.push_back( psSets[0] );
		for( int s = 1; s < SETS; ++s )
			sortedPoints.push_back( PointSet() );

		for( int p = 0; p < POINTS; ++p )
		{
			// Trying to match locations to set 0
			Point pGround = psSets[ 0 ].getPoint( p );

			for( int s = 1; s < SETS; ++s )
			{
				Point pNearest = psSets[ s ].getNearestLinear( pGround );
				sortedPoints[ s ].pushPoint( pNearest );
				psSets[ s ].removePoint( pNearest );
			}
		}

		psSets = sortedPoints;
	}

	#ifdef VISUALIZE_ASCII
	for( int i = 0; i < SETS; ++i )
	{
		char infoLine[1024];
		sprintf( infoLine, "Sample set %d", i + 1, SETS );
		g_infoLine = infoLine;
		clearSampleTable();
		psSets.at( i ).fillPoints( getColorIndex( i ), SAMPLE_SIZE, &g_colorTable[0][0], &g_sampleTable[0][0] );
		draw();
		usleep( 2000000 );
	}
	#endif

	#ifdef VISUALIZE_ASCII
	for( int p = 0; p < POINTS; ++p )
	{
		PointSet indexSet;

		for( int s = 0; s < SETS; ++s )
		{
			indexSet.pushPoint( psSets[ s ].getPoint( p ) );
		}

		char infoLine[1024];
		sprintf( infoLine, "Points at index %d across all sets", p );;
		g_infoLine = infoLine;
			
		clearSampleTable();
		indexSet.fillPoints( getColorIndex( 0 ), SAMPLE_SIZE, &g_colorTable[0][0], &g_sampleTable[0][0] );

		draw();
		usleep( 1000000 );
	}
	#endif

	#ifdef VISUALIZE_ASCII
	usleep( 100000 );
	char infoLine[1024];
	sprintf( infoLine, "ALL SAMPLE SETS" );
	g_infoLine = infoLine;
	clearSampleTable();
	for( int s = 0; s < SETS; ++s )
		psSets.at( s ).fillPoints( getColorIndex( s ), SAMPLE_SIZE, &g_colorTable[0][0], &g_sampleTable[0][0] );
	draw();
	#endif

	#ifdef EXPORT_BIAS
	for( int s = 0; s < SETS; ++s )
		psSets.at( s ).calculateWeights();
	#endif

	#ifndef EXPORT_FILE
	for( int p = 0; p < POINTS; ++p )
	{
		for( int s = 0; s < SETS; ++s )
		{
			Point pPoint = psSets.at( s ).getPoint( p );
			#if 1
			printf( "%Ef, %Ef, \n", pPoint.getXf(), pPoint.getYf() );
			#else
			printf( "Point %d, set %d: X %E, Y %E, bias %E\n", 
					p, s,
					pPoint.getXf() * 2.0f, pPoint.getYf() * 2.0f, pPoint.getWeight() );
			#endif
		}
		printf( "\n" );
	}
	#endif

	#ifdef EXPORT_FILE
	if( std::string( EXPORT_FILE ) != "" )
		writeImage( psSets );
	#endif

	return 0;
}
