/* BSD license throughout
 * Ville Timonen, 2015
 * wili@wili.cc
 */
#include <cstdio>
#include <cstdlib>
#include <cmath>
#include <algorithm>

#include "point.h"
#include "imgwriter.h"

float Point::s_fCenter;
int Point::s_iSampleSize;
int PointSet::s_iSampleSize;

static Point g_pCenter( 0.5f, 0.5f );

Point::Point( int x, int y ) : m_iX( x ), m_iY( y )
{
	m_fX = (float)( x ) / (float)( s_iSampleSize - 1 );
	m_fY = (float)( y ) / (float)( s_iSampleSize - 1 );
}

Point::Point( float x, float y )
{
	m_iX = -1;
	m_iY = -1;
	m_fX = x;
	m_fY = y;
}

void Point::setCenter( float c )
{
	s_fCenter = c;
	g_pCenter.setPos( c, c );
}

void Point::setWeight( float w )
{
	m_fWeight = w;
}

float Point::getWeight()
{
	return m_fWeight;
}

void Point::setSampleSize( int s )
{
	s_iSampleSize = s;
}

void Point::setPos( float x, float y )
{
	m_fX = x;
	m_fY = y;
}

float Point::linearDistanceFromCenter()
{
	float fDistX = m_fX - g_pCenter.getXf();
	float fDistY = m_fY - g_pCenter.getYf();

	return sqrtf( fDistX * fDistX + fDistY * fDistY );
}

float Point::distanceFromCenter()
{
	return distanceFrom( g_pCenter );
}

Point Point::getEdgePoint()
{
	// Vector from the origo to the point
	float vecX = getXf() - g_pCenter.getXf();
	float vecY = getYf() - g_pCenter.getYf();

	// Normalizing
	float fDist = sqrtf( vecX * vecX + vecY * vecY );
	vecX /= fDist;
	vecY /= fDist;

	vecX *= 0.5f;
	vecY *= 0.5f;

	return Point( g_pCenter.getXf() + vecX, g_pCenter.getYf() + vecY );
}

bool Point::isWithinRadius()
{
	// Center is also radius
	return linearDistanceFromCenter() < s_fCenter;
}

int Point::getX() const
{
	return m_iX;
}

int Point::getY() const
{
	return m_iY;
}

float Point::getXf() const
{
	return m_fX;
}

float Point::getYf() const
{
	return m_fY;
}

float Point::distanceFrom( const Point &p )
{
	float thisStep = 0.5f / (float)INTEGRATION_STEPS;
	float stepSize = 1.0f / (float)INTEGRATION_STEPS;

	float fDiffX = p.getXf() - getXf();
	float fDiffY = p.getYf() - getYf();

	float fDiffLength = sqrtf( fDiffX * fDiffX + fDiffY * fDiffY );

	float fStepX = fDiffX * stepSize;
	float fStepY = fDiffY * stepSize;

	float fThisX = getXf() + thisStep * fDiffX;
	float fThisY = getYf() + thisStep * fDiffY;

	float fTotalWeight = 0.0f;
	for( int i = 0; i < INTEGRATION_STEPS; ++i )
	{
		Point pThisStep( fThisX, fThisY );
		float fThisWeight = getDensity( pThisStep.linearDistanceFromCenter() * 2.0f );

		//if( fThisWeight > 0.0f )
			fTotalWeight += fDiffLength * fThisWeight;
		//else
		//	fTotalWeight += 100000.0f * fDiffLength;

		fThisX += fStepX;
		fThisY += fStepY;
	}

	return fTotalWeight;
}

float Point::linearDistanceFrom( const Point &p )
{	
	float fDistX = p.getXf() - getXf();
	float fDistY = p.getYf() - getYf();

	return sqrtf( fDistX * fDistX + fDistY * fDistY );
}

bool Point::operator==( const Point &p )
{
	return m_iX == p.getX() && m_iY == p.getY();
}



PointSet::PointSet() // const int iNumPoints ) : m_iNumPoints( iNumPoints )
{
}

void PointSet::setSampleSize( int s )
{
	s_iSampleSize = s;
}

void PointSet::fillPoints( const int iColor, const int iSampleSize, int *pColorTable, int *pSampleTable )
{
	for( int i = 0; i < m_points.size(); ++i )
	{
		Point &p = m_points[i];
		pColorTable[ p.getX() * iSampleSize + p.getY() ] = iColor;
		char cSymbol;
		if( i < 10 )
			cSymbol = '0' + i;
		else if ( i < 10 + ( 'z' - 'a' ) )
			cSymbol = 'A' + i - 10;
		else if ( i < 10 + ( 'z' - 'a' ) * 2 )
			cSymbol = 'a' + i - ( 10 + ( 'z' - 'a' ) );
		else
			cSymbol = '#';

		pSampleTable[ p.getX() * iSampleSize + p.getY() ] = cSymbol;
	}
}

Point PointSet::addPoint()
{
	float fMaxDistance = 0.0f;
	Point pBest( 0, 0 );

	for( int x = 0; x < s_iSampleSize; ++x )
		for( int y = 0; y < s_iSampleSize; ++y )
		{
			Point pCandidate( x, y );
			//printf( "Created candidate %d %d\n", x, y );
			if( !pCandidate.isWithinRadius() )
				continue;

			// Distance from the center
			float fThisMinDistance = pCandidate.distanceFrom( g_pCenter );

			//printf( "Distance so far %f\n", fThisMinDistance );
			
			// Distance from the edge
			Point pEdge	= pCandidate.getEdgePoint();
			fThisMinDistance = fminf( fThisMinDistance, pCandidate.distanceFrom( pEdge ) ); //0.5f - pCandidate.distanceFromCenter2();
			//printf( "Distance so far %f\n", fThisMinDistance );

			// Distance from all other points
			for( int i = 0; i < m_points.size(); ++i )
				fThisMinDistance = fminf( fThisMinDistance, pCandidate.distanceFrom( m_points.at( i ) ) );

			if( fThisMinDistance > fMaxDistance )
			{
				pBest = pCandidate;
				fMaxDistance = fThisMinDistance;
			}
		}

	/*while( !p.isWithinRadius() )
	{
		int iRandY = rand() % 256;
		int iRandX = rand() % 256;
		p = Point( iRandX, iRandY );
	}*/

	//printf( "Added point %d (at %dx%d)\n", m_points.size(), pBest.getX(), pBest.getY() );

	pushPoint( pBest );
	return pBest;
}

void PointSet::pushPoint( Point p )
{
	m_points.push_back( p );
}

Point PointSet::getPoint( int i ) const
{
	return m_points.at( i );
}

void PointSet::transferPointTo( PointSet *dest )
{
	// We find the next point using poisson disc sampling out of the full set
	float fMaxDistance = 0.0f;

	int iBest = 0;

	for( int i = 0; i < getPointCount(); ++i )
	{
		float fThisMinDistance = 1000.0f;

		// Comparing against every point in the set so far
		for( int s = 0; s < dest->getPointCount(); ++s )
			fThisMinDistance = fminf( fThisMinDistance, dest->getPoint( s ).distanceFrom( getPoint( i ) ) );

		if( fThisMinDistance > fMaxDistance )
		{
			fMaxDistance = fThisMinDistance;
			iBest = i;
		}
	}

	//printf( "Picked point index %d (%d points left, %d points left in dest)\n", iBest, getPointCount(), dest->getPointCount() );
	Point pBest = getPoint( iBest );
	dest->pushPoint( pBest );

	removePoint( pBest );
}

void PointSet::removePoint( Point p )
{
	std::vector< Point >::iterator newEnd = std::remove( m_points.begin(), m_points.end(), p );
	m_points.erase( newEnd, m_points.end() );
}

void PointSet::removePoints()
{
	m_points.clear();
}

Point PointSet::getNearestLinear( Point p )
{
	float fMinDistance = 1000.0f;
	Point pNearest( 0, 0 );

	if( !m_points.size() )
	{
		fprintf( stderr, "No points!\n" );
		exit( 1 );
	}


	for( int i = 0; i < m_points.size(); ++i )
	{
		Point pCandidate = m_points.at( i );

		float fThisDistance = p.linearDistanceFrom( pCandidate );
		if( fThisDistance < fMinDistance )
		{
			fMinDistance = fThisDistance;
			pNearest = pCandidate;
		}
	}

	return pNearest;
}

PointSet PointSet::subSet( int iPoints )
{
	PointSet thisSet;

	if( !iPoints )
		return thisSet;

	thisSet.pushPoint( getPoint( 0 ) );

	while( thisSet.getPointCount() < iPoints )
	{
		// We find the next point using poisson disc sampling out of the full set
		float fMaxDistance = 0.0f;

		int iBest = 0;

		for( int i = 0; i < getPointCount(); ++i )
		{
			float fThisMinDistance = 1000.0f;

			// Comparing against every point in the set so far
			for( int s = 0; s < thisSet.getPointCount(); ++s )
				fThisMinDistance = fminf( fThisMinDistance, thisSet.getPoint( s ).distanceFrom( getPoint( i ) ) );

			if( fThisMinDistance > fMaxDistance )
			{
				fMaxDistance = fThisMinDistance;
				iBest = i;
			}
		}

		thisSet.pushPoint( getPoint( iBest ) );
	}
}

void PointSet::removePoints( const PointSet &p )
{
	std::vector< Point > salvaged;
	
	for( int i = 0; i < m_points.size(); ++i )
	{
		Point thisPoint = m_points.at( i );

		bool bFound = false;
		for( int r = 0; r < p.getPointCount(); ++r )
			if( thisPoint == p.getPoint( r ) )
			{
				bFound = true;
				break;
			}

		if( !bFound )
			salvaged.push_back( thisPoint );
	}

	m_points = salvaged;
}

int PointSet::getPointCount() const
{
	return m_points.size();
}

void PointSet::calculateWeights()
{
	// Calculates PDF-based weight which can be used to remove density distribution bias
	// by multiplying the sample with the weight.
	// Distribute weight such that they sum up to the number of samples.
	float fTotalWeight = 0.0f;

	for( int i = 0; i < m_points.size(); ++i )
	{
		float fThisWeight = 1.0f / getDensity( m_points.at( i ).linearDistanceFromCenter() * 2.0f );
		m_points.at( i ).setWeight( fThisWeight );
		fTotalWeight += fThisWeight;
	}

	float fWeightNormalization = 1.0f / ( fTotalWeight / (float)m_points.size() );

	for( int i = 0; i < m_points.size(); ++i )
	{
		m_points.at( i ).setWeight( m_points.at( i ).getWeight() * fWeightNormalization );
		//printf( "point %d, weight %f, dist from center %f\n", i, m_points.at( i ).getWeight(), m_points.at( i ).linearDistanceFromCenter() );
	}
}
