/*
   BSD license throughout
   Ville Timonen 2013
*/

// This file shows you how the LSAO reference implementation can be used on the high level
#include "config.h"

#include <cstdio>
#include <cstdlib>
#include <unistd.h>
#include <fcntl.h>
#include <string>
#include <sys/stat.h>
#include <sys/mman.h>
#include <cmath>
#include <getopt.h>

#include "sseo.h"
#include "opengl.h"
#include "control.h"

using namespace SSEO;


#define INTERACTIVE
#define SEPARATE_SOURCES // Whether to have depth+normal buffers as separate textures


int main(int argc, char **argv) {
	SSEO::OpenGLController *oglCtrl;
	SSEO::Texture *depthNormalTex, *normalTex, *resultTex;
	
	// Defaults
	int w = 1280;
	int h = 720;
	int borderW = 256;
	int borderH = 144;

	int K = 16;
	float falloffDecay = 4.0f;

	auto gpgpuType =
		#ifdef HAVE_CUDA
		SSEO_CUDA;
		#else
		SSEO_OPENCL;
		#endif

	std::string objFname = "model.obj";

	// Parsing params
	int option;
	while ((option = getopt(argc, argv, "k:f:O:g:b:")) != -1) {
		switch (option) {
			case 'k':
				K = atoi(optarg);
				break;
			case 'f':
				falloffDecay = atof(optarg);
				break;
			case 'b':
				float relBorder;
				relBorder = atof(optarg);
				if (relBorder < 0.0f || relBorder > 1.0f) {
					fprintf(stderr, "You asked me to use %.1f%% borders, which I assume to be a mistake\n", relBorder*100.0f);
					exit(123);
				}
				borderW = (float)w * relBorder + 0.5f;
				borderH = (float)h * relBorder + 0.5f;
				break;
			case 'g':
				int newW, newH;
				sscanf(optarg, "%dx%d", &newW, &newH);
				if (newW < 1e2 || newW > 1e6 || newH < 1e2 || newH > 1e6) {
					fprintf(stderr, "Suspicious dimensions given (%dx%d), aborting\n", newW, newH);
					exit(125);
				}
				w = newW;
				h = newH;
				break;
			case 'O':
				if (std::string(optarg) == "penCL") { // Funnay, I know..
					gpgpuType = SSEO_OPENCL;
					break;
				}
			default:
				fprintf(stderr, "Usage: %s [-kA] [-fB] [-OpenCL] [-gCxD] [-bE] [F]\n", argv[0]);
				exit(124);
		}
	}

	if (optind < argc)
		// We also have a fname incoming..
		objFname = argv[optind];

	try {
		// Normally one would have an existing OpenGL environment and only give SSEO::init() the
		// texture handle to the depth buffer, but in this demo we initialize OpenGL ourselves.
		oglCtrl = new SSEO::OpenGLController(w, borderW, h, borderH);

		#ifdef SEPARATE_SOURCES
		oglCtrl->initRendering(1);
		#else
		oglCtrl->initRendering(2);
		#endif

		oglCtrl->setCamera(0.0f, 0.0f, 0.0f, 0.0f, 0.0f); // Just something to init the projection matrix

		WavefrontObject obj(objFname);
		oglCtrl->loadMesh(obj);
		oglCtrl->render();
		depthNormalTex = oglCtrl->getHeight();

		#ifdef SEPARATE_SOURCES
		normalTex = oglCtrl->getNormal();
		#endif

		resultTex = oglCtrl->getResultTex(w + borderW, h + borderH);

		float projCoefs[] = {
			oglCtrl->projMatrix().elem(0,0),
			oglCtrl->projMatrix().elem(1,1),
			oglCtrl->projMatrix().elem(2,0),
			oglCtrl->projMatrix().elem(2,1)
		};

		SSEO::init(depthNormalTex->getHandle(), 
		#ifdef SEPARATE_SOURCES
				normalTex->getHandle(),
		#else
				0,
		#endif
				resultTex->getHandle(),
				projCoefs, borderW, borderH,
				K, falloffDecay,
				gpgpuType);
	} catch (std::string e) {
		printf("Demo failed: %s\n", e.c_str());
		return -1;
	}



	#ifdef INTERACTIVE
	printf("*** press ESC to QUIT ***\n");
	while (!oglCtrl->abort()) {
	#else
	for (int i = 0; i < 22; ++i) {
	#endif
		#ifdef INTERACTIVE
		oglCtrl->updateCamera();
		#else
		oglCtrl->setCamera(-5.389676, 1.588181, 1.806177, 0.749998, 1.241002);
		#endif

		try {
			oglCtrl->render();

			SSEO::computeLight(true); // True = bench the algorithm after 10 frames
		} catch (std::string e) {
			fprintf(stderr, "Problems with calculating SSAO: %s\n", e.c_str());
			exit(9);
		}

		// Showing the results
		try {
			oglCtrl->blit(resultTex);
		} catch (std::string e) {
			fprintf(stderr, "Problems blitting: %s\n", e.c_str());
		}
	}

	SSEO::uninit();
	delete oglCtrl;

	return 0;
}
