#ifndef _GPGPU_FRAME_H
#define _GPGPU_FRAME_H

#include <string>
#include <vector>

class KernelSrc;
class gpgpuKeywords;

// The idea is that you can reimplement this class for whichever GPGPU system you're using
// Kernel, buffer, and texture handles are re-abstracted as integers
class gpgpuController {
	public:
	gpgpuController(int ctx = 0) { d_keywords = NULL; }
	virtual ~gpgpuController() {}
	virtual int newBuffer(size_t, const void *data = NULL) = 0;
	virtual void fillTexFromBuffer(int, int) = 0;
	virtual void fillBufferFromTex(int, int) { throw std::string("Not implemented"); }
	virtual void clearBuffer(int, void *fourB) = 0;
	virtual void uploadToBuffer(int, void*, size_t) = 0;
	virtual int newKernel(KernelSrc*) = 0;
	virtual void writeKernelSrc(int, std::string) = 0;
	virtual int makeGLTexture(unsigned int) = 0;
	virtual int getBufferFromTex(int) { throw std::string("Not implemented"); }
	virtual int create3DTexture(int w, int h, int d, void*, bool floatData = false) = 0;
	virtual int create2DTextureFromFile(int w, int h, int chans, std::string fName) { throw std::string("Not implemented"); }
	virtual int create2DTextureFromData(int w, int h, int chans, void*) { throw std::string("Not implemented"); }
	virtual int create2DSurface(int w, int h, int chans) { throw std::string("Not implemented"); }
	virtual void nearestSampling(int t) { throw std::string("Not implemented"); }
	virtual void setBufferParam(int, int) = 0;
	virtual void setDataParam(int, size_t, void*) { throw std::string("Not implemented"); }
	virtual void setVar(int, std::string, size_t, void*) { throw std::string("Not implemented"); } // Kernel, variable name, data, size
	virtual void clearKernelParams(int) = 0;
	virtual void *getBufferData(int, size_t&) = 0;
	virtual void *getBufferPtr(int) { throw std::string("Not implemented"); } // Does not have to be implemented
	virtual void setTex(int, int, std::string) = 0;
	virtual void setSurf(int, int, std::string) { throw std::string("Not implemented"); }
	//virtual void setTex3D(int, int, std::string) = 0;
	virtual void setKernelExecConf(int, int totalX, int blockX, int totalY = 1, int blockY = 1) = 0;
	virtual void preferSM(int k) {}; // Prefer shared memory (available in Fermi/Kepler w/ CUDA)
	virtual void executeKernel(int) = 0;
	virtual void reportTiming(int) = 0; // Reports execution times for a test run
	gpgpuKeywords *getKeywords() { if (!d_keywords) throw std::string("No keywords initted yet"); return d_keywords; }		

	protected:
	gpgpuKeywords *d_keywords;
};


// This acts as a dictionary, such that the KernelSrc does not have
// to know the underlying language.
class gpgpuKeywords {
	public:
	gpgpuKeywords() {}
	virtual std::string blockX() { throw std::string("Unimplemented keyword 1"); }
	virtual std::string blockY() { throw std::string("Unimplemented keyword 2"); }
	virtual std::string threadX() { throw std::string("Unimplemented keyword 3"); }
	virtual std::string threadY() { throw std::string("Unimplemented keyword 4"); }
	virtual std::string blockDimX() { throw std::string("Unimplemented keyword 5"); }
	virtual std::string blockDimY() { throw std::string("Unimplemented keyword 6"); }
	virtual std::string globalThreadX() { throw std::string("Unimplemented keyword 7"); }
	virtual std::string globalThreadY() { throw std::string("Unimplemented keyword 8"); }
	virtual std::string sharedMem() { throw std::string("Unimplemented keyword 9"); }
	virtual std::string constMem() { throw std::string("Unimplemented keyword 10"); }
	virtual std::string localSync() { throw std::string("Unimplemented keyword 11"); }
	virtual std::string halfType() { throw std::string("Unimplemented keyword 44"); }
	virtual std::string regType() { throw std::string("Unimplemented keyword 45"); }
	virtual std::string sharedType() { throw std::string("Unimplemented keyword 46"); }
	virtual std::string globalType() { throw std::string("Unimplemented keyword 47"); }
	virtual std::string kernelDecl(std::string name, std::string params) { throw std::string("Unimplemented keyword 12"); }
	virtual std::string funcDecl(std::string name, std::string ret, std::string params, std::vector<std::string> tex = std::vector<std::string>()) { throw std::string("Unimplemented keyword 13"); }
	virtual std::string float2Operators() { throw std::string("Unimplemented keyword 14"); }
	virtual std::string float3Operators() { throw std::string("Unimplemented keyword 15"); }
	virtual std::string vectorOperators() { throw std::string("unimplemented keyword 16"); }
	virtual std::string float2Ctor(std::string valX, std::string valY) { throw std::string("unimplemented keyword 17"); }
	virtual std::string readHalf2(std::string dest, std::string src, std::string ptr = "", std::string offset = "0") { throw std::string("unimplemented keyword 18"); } // ptr defines the source type:  "" = register, "reg" = register pointer, "shared" = shared memory pointer, "global" = global memory pointer.  Offset from the pointer, if pointer is set

	virtual std::string writeHalf2(std::string dest, std::string src, std::string ptr = "", std::string offset = "0") { throw std::string("unimplemented keyword 19"); }
	virtual std::string writeHalf(std::string dest, std::string src) { throw std::string("unimplemented keyword 48"); }

	virtual std::string float3Ctor(std::string valX, std::string valY, std::string valZ) { throw std::string("unimplemented keyword 20"); }
	virtual std::string float4Ctor(std::string valX, std::string valY, std::string valZ, std::string valW) { throw std::string("unimplemented keyword 21"); }
	virtual std::string div(std::string, std::string) { throw std::string("unimplemented keyword 22"); }
	virtual std::string rcp(std::string) { throw std::string("unimplemented keyword 23"); }
	virtual std::string floorf(std::string) { throw std::string("unimplemented keyword 24"); }
	virtual std::string sqrt(std::string) { throw std::string("unimplemented keyword 25"); }
	virtual std::string rsqrt(std::string) { throw std::string("unimplemented keyword 26"); }
	virtual std::string pow(std::string, std::string) { throw std::string("unimplemented keyword 27"); }
	virtual std::string sin(std::string) { throw std::string("unimplemented keyword 28"); }
	virtual std::string cos(std::string) { throw std::string("unimplemented keyword 29"); }
	virtual std::string sincos(std::string angle, std::string sintarget, std::string costarget) { throw std::string("unimplemented keyword 30"); }
	virtual std::string exp2(std::string) { throw std::string("unimplemented keyword 31"); }
	virtual std::string maxf(std::string, std::string) { throw std::string("unimplemented keyword 32"); }
	virtual std::string minf(std::string, std::string) { throw std::string("unimplemented keyword 33"); }
	virtual std::string absf(std::string) { throw std::string("unimplemented keyword 34"); }
	virtual std::string tex2DSample1(std::string id, std::string coordX, std::string coordY) { throw std::string("unimplemented keyword 35"); }
	virtual std::string tex2DSample4(std::string id, std::string coordX, std::string coordY) { throw std::string("unimplemented keyword 36"); }
	virtual std::string tex3DSample(std::string id, std::string coordX, std::string coordY, std::string coordZ) { throw std::string("unimplemented keyword 37"); }
	virtual std::string surf2DWrite(std::string id, std::string coordX, std::string coordY, std::string type, std::string value) { throw std::string("Unimplemented keyword 38"); }
	virtual std::string surf2DRead(std::string id, std::string coordX, std::string coordY, std::string type) { throw std::string("Unimplemented keyword 39"); }

	// And the ones that take parameters
	virtual std::string atomicMin(std::string dest, std::string value) { throw std::string("Unimplemented keyword 40"); }
	virtual std::string atomicMax(std::string dest, std::string value) { throw std::string("Unimplemented keyword 41"); }
	virtual std::string atomicAdd(std::string dest, std::string value) { throw std::string("Unimplemented keyword 42"); }

	// In case you need to declare/#define stuff for every source
	virtual std::string header(bool h = true) { return std::string(""); }

	virtual std::string fileExt() { throw std::string("Unimplemented keyword 43"); }

	bool replaceContent(std::string&); // Returns false if we didn't recognize the content, true otherwise.  Modifies the parameter otherwise


	/*virtual std::string tex2DDeclFloat(std::string) { throw std::string("unimplemented keyword"); }
	virtual std::string tex2DDeclFloat4(std::string) { throw std::string("unimplemented keyword"); }
	virtual std::string tex3DDeclFloat(std::string) { throw std::string("unimplemented keyword"); }*/

	private:

	std::string getNextParam(std::string, size_t);
};

class KernelSrc {
	public:
	KernelSrc();
	void setKeywords(gpgpuKeywords*);
	
	std::string getSrc();
	std::string getAsm();
	std::string getAsmFunc();
	void append(const char*, ...);
	void append(const std::string str);
	void appendFromFile(const std::string fname);
	void incrementIndent(int steps = 1);
	void decrementIndent(int steps = 1);

	void useAsm(const std::string fname);

	void write(const std::string fname); // Write the sources out
	static void write(const std::string fname, const std::string src);

	void replace(std::string, std::string);

	static std::string genStr(const char*, ...);
	static std::string readFile(const std::string fname);

	protected:
	std::string d_preExpandSrc;
	std::string d_postExpandSrc;
	std::string d_uploadAsm;
	gpgpuKeywords *d_keywords;

	int d_indent;

	virtual void genSrc() {} // You can inherit this class and fill this in (in that case, don't append anything externally)
	
	private:
	void expandKeywords(); // Go through the pseudotags and replace using gpgpuKeywords
};

#endif // _GPGPU_FRAME_H
