#include "hostTypes.h"
#include <cmath>
#include <cstdio>


using namespace SSEO;

float2::float2(float nx, float ny) : x(nx), y(ny) {}

float2::operator double2() {
	return double2((double)x, (double)y);
}

float2 float2::operator+(const float2 add) {
	return float2(x + add.x, y + add.y);
}

float2 float2::operator-(const float2 sub) {
	return float2(x - sub.x, y - sub.y);
}

float2 float2::operator-() {
	return float2(-x, -y);
}

void float2::operator+=(const float2 add) {
	x += add.x;
	y += add.y;
}

void float2::operator-=(const float2 add) {
	x -= add.x;
	y -= add.y;
}

float2 float2::operator*(const float c) {
	return float2(x*c, y*c);
}

bool float2::operator==(const float2 comp) {
	return x == comp.x && y == comp.y;
}

bool float2::operator!=(const float2 comp) {
	return !(*this == comp);
}

void float2::operator*=(const float c) {
	x *= c;
	y *= c;
}

void float2::operator*=(const float2 c) {
	x *= c.x;
	y *= c.y;
}

double2::double2(double nx, double ny) : x(nx), y(ny) {}

double2 double2::operator+(const double2 add) {
	return double2(x + add.x, y + add.y);
}

double2 double2::operator-(const double2 sub) {
	return double2(x - sub.x, y - sub.y);
}

double2 double2::operator-() {
	return double2(-x, -y);
}

void double2::operator+=(const double2 add) {
	x += add.x;
	y += add.y;
}

void double2::operator-=(const double2 add) {
	x -= add.x;
	y -= add.y;
}

double2 double2::operator*(const double c) {
	return double2(x*c, y*c);
}

double2::operator float2() {
	return float2((float)x, (float)y);
}

bool double2::operator==(const double2 comp) {
	return x == comp.x && y == comp.y;
}

bool double2::operator!=(const double2 comp) {
	return !(*this == comp);
}

void double2::operator*=(const double c) {
	x *= c;
	y *= c;
}

float double2::length() {
	return sqrt(x*x + y*y);
}

bool float2::lowerX(const float2 a, const float2 b) {
	return a.x < b.x;
}

float float2::dot(const float2 b) {
	return x*b.x + y*b.y;
}


Matrix4::Matrix4() {
	for (int x = 0; x < 4; ++x)
		for (int y = 0; y < 4; ++y)
			elem(x, y) = (x == y) ? 1.0f : 0.0f;
}

Matrix4::Matrix4(const Matrix4 &c) {
	for (int y = 0; y < 4; ++y)
		for (int x = 0; x < 4; ++x)
			elem(x, y) = c.elem(x, y);
}

float *Matrix4::data() {
	return d_data;
}

float *Matrix4::colMajorData() {
	// Filling the colmajordata
	for (int y = 0; y < 4; ++y)
		for (int x = 0; x < 4; ++x)
			d_colMajorData[x*4 + y] = d_data[y*4 + x];
	return d_colMajorData;
}

float Matrix4::determinant() {
	return elem(3,0)*elem(2,1)*elem(1,2)*elem(0,3) - elem(2,0)*elem(3,1)*elem(1,2)*elem(0,3) - elem(3,0)*elem(1,1)*elem(2,2)*elem(0,3) + elem(1,0)*elem(3,1)*elem(2,2)*elem(0,3)+ elem(2,0)*elem(1,1)*elem(3,2)*elem(0,3) - elem(1,0)*elem(2,1)*elem(3,2)*elem(0,3) - elem(3,0)*elem(2,1)*elem(0,2)*elem(1,3) + elem(2,0)*elem(3,1)*elem(0,2)*elem(1,3)+ elem(3,0)*elem(0,1)*elem(2,2)*elem(1,3) - elem(0,0)*elem(3,1)*elem(2,2)*elem(1,3) - elem(2,0)*elem(0,1)*elem(3,2)*elem(1,3) + elem(0,0)*elem(2,1)*elem(3,2)*elem(1,3)+ elem(3,0)*elem(1,1)*elem(0,2)*elem(2,3) - elem(1,0)*elem(3,1)*elem(0,2)*elem(2,3) - elem(3,0)*elem(0,1)*elem(1,2)*elem(2,3) + elem(0,0)*elem(3,1)*elem(1,2)*elem(2,3)+ elem(1,0)*elem(0,1)*elem(3,2)*elem(2,3) - elem(0,0)*elem(1,1)*elem(3,2)*elem(2,3) - elem(2,0)*elem(1,1)*elem(0,2)*elem(3,3) + elem(1,0)*elem(2,1)*elem(0,2)*elem(3,3)+ elem(2,0)*elem(0,1)*elem(1,2)*elem(3,3) - elem(0,0)*elem(2,1)*elem(1,2)*elem(3,3) - elem(1,0)*elem(0,1)*elem(2,2)*elem(3,3) + elem(0,0)*elem(1,1)*elem(2,2)*elem(3,3);
}

void Matrix4::invert() {
	float coef = 1.0f/determinant();

	float newData[] = {
		elem(2,1)*elem(3,2)*elem(1,3) - elem(3,1)*elem(2,2)*elem(1,3) + elem(3,1)*elem(1,2)*elem(2,3) - elem(1,1)*elem(3,2)*elem(2,3) - elem(2,1)*elem(1,2)*elem(3,3) + elem(1,1)*elem(2,2)*elem(3,3),
		elem(3,0)*elem(2,2)*elem(1,3) - elem(2,0)*elem(3,2)*elem(1,3) - elem(3,0)*elem(1,2)*elem(2,3) + elem(1,0)*elem(3,2)*elem(2,3) + elem(2,0)*elem(1,2)*elem(3,3) - elem(1,0)*elem(2,2)*elem(3,3),
		elem(2,0)*elem(3,1)*elem(1,3) - elem(3,0)*elem(2,1)*elem(1,3) + elem(3,0)*elem(1,1)*elem(2,3) - elem(1,0)*elem(3,1)*elem(2,3) - elem(2,0)*elem(1,1)*elem(3,3) + elem(1,0)*elem(2,1)*elem(3,3),
		elem(3,0)*elem(2,1)*elem(1,2) - elem(2,0)*elem(3,1)*elem(1,2) - elem(3,0)*elem(1,1)*elem(2,2) + elem(1,0)*elem(3,1)*elem(2,2) + elem(2,0)*elem(1,1)*elem(3,2) - elem(1,0)*elem(2,1)*elem(3,2),
		elem(3,1)*elem(2,2)*elem(0,3) - elem(2,1)*elem(3,2)*elem(0,3) - elem(3,1)*elem(0,2)*elem(2,3) + elem(0,1)*elem(3,2)*elem(2,3) + elem(2,1)*elem(0,2)*elem(3,3) - elem(0,1)*elem(2,2)*elem(3,3),
		elem(2,0)*elem(3,2)*elem(0,3) - elem(3,0)*elem(2,2)*elem(0,3) + elem(3,0)*elem(0,2)*elem(2,3) - elem(0,0)*elem(3,2)*elem(2,3) - elem(2,0)*elem(0,2)*elem(3,3) + elem(0,0)*elem(2,2)*elem(3,3),
		elem(3,0)*elem(2,1)*elem(0,3) - elem(2,0)*elem(3,1)*elem(0,3) - elem(3,0)*elem(0,1)*elem(2,3) + elem(0,0)*elem(3,1)*elem(2,3) + elem(2,0)*elem(0,1)*elem(3,3) - elem(0,0)*elem(2,1)*elem(3,3),
		elem(2,0)*elem(3,1)*elem(0,2) - elem(3,0)*elem(2,1)*elem(0,2) + elem(3,0)*elem(0,1)*elem(2,2) - elem(0,0)*elem(3,1)*elem(2,2) - elem(2,0)*elem(0,1)*elem(3,2) + elem(0,0)*elem(2,1)*elem(3,2),
		elem(1,1)*elem(3,2)*elem(0,3) - elem(3,1)*elem(1,2)*elem(0,3) + elem(3,1)*elem(0,2)*elem(1,3) - elem(0,1)*elem(3,2)*elem(1,3) - elem(1,1)*elem(0,2)*elem(3,3) + elem(0,1)*elem(1,2)*elem(3,3),
		elem(3,0)*elem(1,2)*elem(0,3) - elem(1,0)*elem(3,2)*elem(0,3) - elem(3,0)*elem(0,2)*elem(1,3) + elem(0,0)*elem(3,2)*elem(1,3) + elem(1,0)*elem(0,2)*elem(3,3) - elem(0,0)*elem(1,2)*elem(3,3),
		elem(1,0)*elem(3,1)*elem(0,3) - elem(3,0)*elem(1,1)*elem(0,3) + elem(3,0)*elem(0,1)*elem(1,3) - elem(0,0)*elem(3,1)*elem(1,3) - elem(1,0)*elem(0,1)*elem(3,3) + elem(0,0)*elem(1,1)*elem(3,3),
		elem(3,0)*elem(1,1)*elem(0,2) - elem(1,0)*elem(3,1)*elem(0,2) - elem(3,0)*elem(0,1)*elem(1,2) + elem(0,0)*elem(3,1)*elem(1,2) + elem(1,0)*elem(0,1)*elem(3,2) - elem(0,0)*elem(1,1)*elem(3,2),
		elem(2,1)*elem(1,2)*elem(0,3) - elem(1,1)*elem(2,2)*elem(0,3) - elem(2,1)*elem(0,2)*elem(1,3) + elem(0,1)*elem(2,2)*elem(1,3) + elem(1,1)*elem(0,2)*elem(2,3) - elem(0,1)*elem(1,2)*elem(2,3),
		elem(1,0)*elem(2,2)*elem(0,3) - elem(2,0)*elem(1,2)*elem(0,3) + elem(2,0)*elem(0,2)*elem(1,3) - elem(0,0)*elem(2,2)*elem(1,3) - elem(1,0)*elem(0,2)*elem(2,3) + elem(0,0)*elem(1,2)*elem(2,3),
		elem(2,0)*elem(1,1)*elem(0,3) - elem(1,0)*elem(2,1)*elem(0,3) - elem(2,0)*elem(0,1)*elem(1,3) + elem(0,0)*elem(2,1)*elem(1,3) + elem(1,0)*elem(0,1)*elem(2,3) - elem(0,0)*elem(1,1)*elem(2,3),
		elem(1,0)*elem(2,1)*elem(0,2) - elem(2,0)*elem(1,1)*elem(0,2) + elem(2,0)*elem(0,1)*elem(1,2) - elem(0,0)*elem(2,1)*elem(1,2) - elem(1,0)*elem(0,1)*elem(2,2) + elem(0,0)*elem(1,1)*elem(2,2)
	};

	for (int i = 0; i < 16; ++i)
		d_data[i] = newData[i]*coef;
}

void Matrix4::translate(float x, float y, float z) {
	d_data[3] = x;
	d_data[3+4] = y;
	d_data[3+8] = z;
}

void Matrix4::rotate(float angle, float x, float y, float z) {
	float coef = 1.0f/sqrtf(x*x + y*y + z*z);
	x *= coef;
	y *= coef;
	z *= coef;

	float c = cosf(angle);
	float s = sinf(angle);
	d_data[0] = x*x*(1-c)+c;
	d_data[1] = x*y*(1-c)-z*s;
	d_data[2] = x*z*(1-c)+y*s;
	d_data[3] = 0;
	d_data[4] = y*x*(1-c)+z*s;
	d_data[5] = y*y*(1-c)+c;
	d_data[6] = y*z*(1-c)-x*s;
	d_data[7] = 0;
	d_data[8] = x*z*(1-c)-y*s;
	d_data[9] = y*z*(1-c)+x*s;
	d_data[10] = z*z*(1-c)+c; 
	d_data[11] = 0;
	d_data[12] = 0;
	d_data[13] = 0;
	d_data[14] = 0;
	d_data[15] = 1;
}

void Matrix4::perspective(float left, float right, float bottom, float top, float nearVal, float farVal) {
	float A = (right+left)/(right-left);
	float B = (top+bottom)/(top-bottom);
	float C = -(farVal+nearVal)/(farVal-nearVal);
	float D = -(2.0f*farVal*nearVal)/(farVal-nearVal);

	d_data[0] = (2.0f*nearVal)/(right-left);
	d_data[1] = 0;
	d_data[2] = A;
	d_data[3] = 0;
	d_data[4] = 0;
	d_data[5] = (2.0f*nearVal)/(top-bottom);
	d_data[6] = B;
	d_data[7] = 0;
	d_data[8] = 0; 
	d_data[9] = 0; 
	d_data[10] = C; 
	d_data[11] = D;
	d_data[12] = 0; 
	d_data[13] = 0; 
	d_data[14] = -1; 
	d_data[15] = 0;
}

const float Matrix4::elem(int x, int y) const {
	return d_data[y*4 + x];
}

float &Matrix4::elem(int x, int y) {
	return d_data[y*4 + x];
}

Matrix4 Matrix4::operator*(const Matrix4 B) {
	Matrix4 result;
	for (int y = 0; y < 4; ++y)
		for (int x = 0; x < 4; ++x) {
			result.elem(x, y) = 0.0f;
			for (int k = 0; k < 4; ++k)
				result.elem(x, y) += elem(k, y) * B.elem(x, k);
		}

	return result;
}

Matrix4 &Matrix4::operator=(const Matrix4 &n) {
	for (int x = 0; x < 4; ++x)
		for (int y = 0; y < 4; ++y)
			elem(x,y) = n.elem(x,y);
	/*for (int i = 0; i < 16; ++i) {
		data()[i] = n.data()[i];
		colMajorData()[i] = n.colMajorData()[i];
	}*/

	return *this;
}

void Matrix4::print() {
	for (int y = 0; y < 4; ++y) {
		for (int x = 0; x < 4; ++x)
			printf("%E\t", elem(y, x));
		printf("\n");
	}
}


Matrix3::Matrix3(const Matrix4 m4) {
	d_data[0] = m4.elem(0, 0);
	d_data[1] = m4.elem(1, 0);
	d_data[2] = m4.elem(2, 0);
	d_data[3] = m4.elem(0, 1);
	d_data[4] = m4.elem(1, 1);
	d_data[5] = m4.elem(2, 1);
	d_data[6] = m4.elem(0, 2);
	d_data[7] = m4.elem(1, 2);
	d_data[8] = m4.elem(2, 2);
}

float *Matrix3::colMajorData() {
	// Filling the colmajordata
	for (int y = 0; y < 3; ++y)
		for (int x = 0; x < 3; ++x)
			d_colMajorData[x*3 + y] = d_data[y*3 + x];
	return d_colMajorData;
}

float *Matrix3::data() {
	return d_data;
}


float3::float3(float x, float y, float z) : 
	d_x(x), d_y(y), d_z(z) {}

float float3::x() const { return d_x; }
float float3::y() const { return d_y; }
float float3::z() const { return d_z; }

void float3::x(float v) { d_x = v; }
void float3::y(float v) { d_y = v; }
void float3::z(float v) { d_z = v; }

void float3::operator+=(const float3 add) {
	d_x += add.x();
	d_y += add.y();
	d_z += add.z();
}

float3 float3::operator-(const float3 sub) {
	return float3(x()-sub.x(), y()-sub.y(), z()-sub.z());
}

void float3::normalize() {
	float coef = 1.0f/sqrtf(x()*x() + y()*y() + z()*z());
	d_x *= coef;
	d_y *= coef;
	d_z *= coef;
}

void float3::normalize(float &x, float &y, float &z) {
	float coef = 1.0f/sqrt(x*x + y*y + z*z);
	x *= coef;
	y *= coef;
	z *= coef;
}

float3 float3::cross(const float3 b) {
	return float3(
			y()*b.z() - z()*b.y(),
			z()*b.x() - x()*b.z(),
			x()*b.y() - y()*b.x());
}
