#include "opengl.h"

#include <cstdio>
#include <cstdlib>
#include <fstream>
#include <string.h>
#include <X11/Xatom.h>
#include <math.h>


using namespace SSEO;

ShaderPool *SSEO::g_defaultShaderPool = NULL;

OpenGLController::OpenGLController(int w, int borderW, int h, int borderH, std::string Xdpy, int posX, int posY) : 
		d_winWidth(w+borderW), d_winHeight(h+borderH), 
		d_borderFracW((float)borderW/(float)(w+borderW)), d_borderFracH((float)borderH/(float)(h+borderH)),
		d_blitTex(NULL), d_renderShaderData(NULL), d_abort(false),
	   	d_meshPolys(0), d_heightTex(NULL), d_normTex(NULL), d_normShaderData(NULL) {

	if (!(d_dpy = XOpenDisplay(Xdpy.c_str()))) { // Has to be supplied, always
		throw std::string("Couldn't open X11 display\n");
	}

	int attr[] = {
		GLX_RGBA,
		GLX_RED_SIZE, 1,
		GLX_GREEN_SIZE, 1,
		GLX_BLUE_SIZE, 1,
		GLX_DOUBLEBUFFER,
		GLX_DEPTH_SIZE, 1,
		None
	};

	int scrnum = DefaultScreen(d_dpy);
	Window root = RootWindow(d_dpy, scrnum);
    
	// We need this for OpenGL 3+
	int elemc;
	GLXFBConfig *fbcfg = glXChooseFBConfig(d_dpy, scrnum, NULL, &elemc);
	if (!fbcfg) {
		throw std::string("Couldn't get FB configs\n");
	} else
		printf("Got %d FB configs\n", elemc);

	XVisualInfo *visinfo = glXChooseVisual(d_dpy, scrnum, attr);

	if (!visinfo) {
		throw std::string("Couldn't get a visual\n");
	}

	// Window parameters
	XSetWindowAttributes winattr;
	winattr.background_pixel = 0;
	winattr.border_pixel = 0;
	winattr.colormap = XCreateColormap(d_dpy, root, visinfo->visual, AllocNone);
	winattr.event_mask = StructureNotifyMask | ExposureMask | KeyPressMask;
	unsigned long mask = CWBackPixel | CWBorderPixel | CWColormap | CWEventMask;

	printf("Window depth %d, w %d h %d\n", visinfo->depth, d_winWidth, d_winHeight);
	d_win = XCreateWindow(d_dpy, root, posX, posY, d_winWidth, d_winHeight, 0, 
			visinfo->depth, InputOutput, visinfo->visual, mask, &winattr);

	// OpenGL version
	int gl3attr[] = {
        GLX_CONTEXT_MAJOR_VERSION_ARB, 4,
        GLX_CONTEXT_MINOR_VERSION_ARB, 1,
        //GLX_CONTEXT_FLAGS_ARB, GLX_CONTEXT_FORWARD_COMPATIBLE_BIT_ARB,
		None
    };

	d_ctx = glXCreateContextAttribsARB(d_dpy, fbcfg[0], NULL, true, gl3attr);

	if (!d_ctx)
		throw std::string("Couldn't create an OpenGL context\n");

	XFree(visinfo);

	// Setting the window name
	XTextProperty windowName;
	windowName.value = (unsigned char *) "SSFFAO";
	windowName.encoding = XA_STRING;
	windowName.format = 8;
	windowName.nitems = strlen((char *) windowName.value);

	XSetWMName(d_dpy, d_win, &windowName);

	XMapWindow(d_dpy, d_win);
	glXMakeCurrent(d_dpy, d_win, d_ctx);
	
	printf("OpenGL:\n\tvendor %s\n\trenderer %s\n\tversion %s\n\tshader language %s\n", //\textensions %s\n",
			glGetString(GL_VENDOR), glGetString(GL_RENDERER), glGetString(GL_VERSION),
			glGetString(GL_SHADING_LANGUAGE_VERSION));//, glGetString(GL_EXTENSIONS));

	g_defaultShaderPool = new ShaderPool();

	// Telling X to give us pointer events
	XSelectInput(d_dpy, d_win, PointerMotionMask|ButtonPressMask|ButtonReleaseMask|KeyPressMask);

	//hideCursor();

	for (int i = 0; i < 5; ++i)
		d_cameraParams.push_back(0.0f);
	d_buttonPressed = false;
}

OpenGLController::~OpenGLController() {
	// FIXME As this is only for demo testing, we're not doing any nice cleaning, we let it collapse on its own
}

float *OpenGLController::getFBData(size_t *size) {
	size_t actSize = d_winWidth * d_winHeight * sizeof(float) * 4;
	if (size)
		*size = actSize;

	float *data = (float*) malloc(actSize);
	glReadPixels(0, 0, d_winWidth, d_winHeight, GL_RGBA, GL_FLOAT, data);
	checkErrors("Export FB");

	return data;
}

void OpenGLController::vomitFB(std::string fname) {
	FILE *out = fopen(fname.c_str(), "w+");
	size_t size;

	float *data = getFBData(&size);

	//printf("Trying to write %ld bytes\n", size);
	size_t written = fwrite(data, sizeof(float), size/sizeof(float), out);
	printf("Wrote %ld items (%ld bytes each)\n", written, sizeof(float));
	fclose(out);
}

void OpenGLController::checkErrors(std::string msg) {
	GLuint er = glGetError();
	if (er) {
		char glmsg[1024];
		sprintf(glmsg, (msg == "") ? "OpenGL error 0x%x" : ": OpenGL error 0x%x", er);
		msg += std::string(glmsg);
		throw msg;
	}
}

int OpenGLController::getTexParam(GLuint tex, GLenum pname) {
	GLint prevTex;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &prevTex);
	glBindTexture(GL_TEXTURE_2D, tex);

	GLint data;
	glGetTexLevelParameteriv(GL_TEXTURE_2D, 0, pname, &data);

	glBindTexture(GL_TEXTURE_2D, prevTex);
	checkErrors("getTexParam");

	return data;
}

Texture *OpenGLController::getTestHF(float *data) {
	GLuint texHandle;
	glGenTextures(1, &texHandle);
	GLint prevTex;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &prevTex);
	glBindTexture(GL_TEXTURE_2D, texHandle);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_R32F, d_winWidth, d_winHeight, 0, GL_RED, GL_FLOAT, data);
	glBindTexture(GL_TEXTURE_2D, prevTex);
	checkErrors("Get test height field");
	
	return d_heightTex = new Texture(GL_TEXTURE_2D, texHandle, "height");
}

Texture *OpenGLController::getResultTex(int w, int h) {
	GLuint texHandle;
	glGenTextures(1, &texHandle);
	GLint prevTex;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &prevTex);
	glBindTexture(GL_TEXTURE_2D, texHandle);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	//glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32F, w, h, 0, GL_RGBA, GL_FLOAT, NULL);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_R32F, w, h, 0, GL_RED, GL_FLOAT, NULL);
	glBindTexture(GL_TEXTURE_2D, prevTex);
	checkErrors("Get result color buffer");
	
	return new Texture(GL_TEXTURE_2D, texHandle, "result");
}

Texture *OpenGLController::genNormalFromHeight() {
	if (!d_heightTex)
		throw std::string("Height not generated");
	if (d_normTex && !d_normShaderData)
		throw std::string("You already decided to render normal with height");

	// We're initting the normal rendering if here for the first time
	if (!d_normTex) {
		glGenFramebuffers(1, &d_normalFbo);
		glBindFramebuffer(GL_DRAW_FRAMEBUFFER, d_normalFbo);
		checkErrors("Normal FBO gen");
		
		GLuint normTex;
		glGenTextures(1, &normTex);

		d_normTex = new Texture(GL_TEXTURE_2D, normTex, "normal texture", d_depthWidth, d_depthHeight);

		d_normTex->bind();
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32F, d_depthWidth, d_depthHeight, 0, GL_RGBA, GL_FLOAT, NULL);
		glFramebufferTexture2D(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, normTex, 0);

		checkErrors("normal tex gen");

		Shader *s = g_defaultShaderPool->getShader("normal");
		s->activate();
		d_normShaderData = new ShaderData(s);
		d_normShaderData->addTexture("heightTex", d_heightTex);
		GLfloat texCoord[] = { 
			0.0f, 1.0f, 
			0.0f, 0.0f,
			1.0f, 1.0f,
			1.0f, 0.0f
		};
		d_normShaderData->addVData("texCoord", GL_FLOAT, 2, 4, texCoord);
		d_normShaderData->addVData("pos", GL_FLOAT, 2, 4, texCoord);

		// Then setting the shader's own data..
		GLint stepLoc = glGetUniformLocation(s->handle(), "stepVec");
		if (stepLoc == -1)
			throw std::string("Couldn't find step");
		// d_depthWidth or d_depthWidth-1?  Doesn't matter if we mess this up,
		// the difference is tiny and you're supposed to supply your own "properly generated"
		// normal map anyway.  This is just a demo
		glUniform2f(stepLoc, 1.0f/(float)d_depthWidth, 1.0f/(float)d_depthHeight);
		checkErrors("set init norm shaderdata");
	}

	glBindFramebuffer(GL_DRAW_FRAMEBUFFER, d_normalFbo);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
	glDisable(GL_DEPTH_TEST);
	glViewport(0, 0, d_depthWidth, d_depthHeight);
	d_normShaderData->shader()->activate();
	d_normShaderData->bind();

	glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
	checkErrors("Render normal");

	return d_normTex;
}

Texture *OpenGLController::getHeight() {
	if (!d_heightTex)
		throw std::string("Can't get height since rendering isn't initialized");
	return d_heightTex;
}

Texture *OpenGLController::getNormal() {
	if (!d_normTex)
		throw std::string("Can't get normal since it's not done");
	return d_normTex;
}

void OpenGLController::render() {
	if (!d_renderShaderData)
		throw std::string("Init rendering first");

	if (!d_meshPolys)
		throw std::string("Load an object first");

	glBindFramebuffer(GL_DRAW_FRAMEBUFFER, d_renderFbo);
	checkErrors("Binding render FBO");
	glEnable(GL_DEPTH_TEST);
	glViewport(0, 0, d_depthWidth, d_depthHeight);
	glClearDepth(1.0);
	glClear(GL_DEPTH_BUFFER_BIT);

	// Also custom-clearing the depth tex (only necessary with scenes that have holes)
	glBindBuffer(GL_PIXEL_UNPACK_BUFFER, d_zeroBuffer);
	glBindTexture(GL_TEXTURE_2D, d_heightTex->getHandle());

	if (d_normTex)
		glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, d_depthWidth, d_depthHeight, GL_RED, GL_FLOAT, 0);
	else
		glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, d_depthWidth, d_depthHeight, GL_RGBA, GL_FLOAT, 0);

	checkErrors("Resetting height tex");

	glBindTexture(GL_TEXTURE_2D, 0);
	glBindBuffer(GL_PIXEL_UNPACK_BUFFER, 0);
	

	d_renderShaderData->shader()->activate();
    d_renderShaderData->bind();

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, d_indexBuffer);
    //glDrawArrays(GL_TRIANGLES, 0, 3*d_meshPolys);
	//printf("Trying to render %d triangles\n", d_meshPolys);

	static const GLenum renderTargets[] = {
		GL_COLOR_ATTACHMENT0,
		GL_COLOR_ATTACHMENT1
	};

	if (d_renderNormal) {
		if (d_normTex)
			glDrawBuffers(2, renderTargets);
		else 
			glDrawBuffers(1, renderTargets);
	}

	glDrawElements(GL_TRIANGLES, d_meshPolys*3, GL_UNSIGNED_INT, NULL);

	if (d_renderNormal)
		glDrawBuffers(1, renderTargets);

	checkErrors("At the end of render");
	glBindFramebuffer(GL_DRAW_FRAMEBUFFER, 0);
}

bool OpenGLController::abort() {
	return d_abort;
}

void OpenGLController::updateCamera() {
	const int cameraRail = 0; // -1 = record, 1 = play, 0 = interactive

	if (cameraRail == 0 || cameraRail == -1) {
		XEvent ev;
		int zeroX = d_winWidth/2;
		int zeroY = d_winHeight/2;

		while (XPending(d_dpy) > 0) {
			XNextEvent(d_dpy, &ev);
			/*printf("Got event of type %d\n", ev.type);
			if (ev.type == 6)
				printf("x %d y %d\n", ((XMotionEvent*)&ev)->x, ((XMotionEvent*)&ev)->y);
			if (ev.type == 5)
				printf("x %d y %d\n", ((XMotionEvent*)&ev)->x, ((XMotionEvent*)&ev)->y);
			if (ev.type == 4)
				printf("x %d y %d\n", ((XMotionEvent*)&ev)->x, ((XMotionEvent*)&ev)->y);*/
			
			int newX = ((XMotionEvent*)&ev)->x;
			int newY = ((XMotionEvent*)&ev)->y;
			float coef = 0.001f;

			if (ev.type == 6) {
				d_cameraParams.at(4) += (float)(newX - zeroX)*coef;
				d_cameraParams.at(3) += (float)(newY - zeroY)*coef;
				//newX, newY;
			}
			if (ev.type == 5) {
				d_buttonPressed = false;
			} 
			if (ev.type == 4) {
				d_buttonPressed = true;
			}

			if (ev.type == KeyPress) {
				XKeyEvent *kev = (XKeyEvent*)&ev;
				if (kev->keycode == XKeysymToKeycode(d_dpy, XK_Escape))
					d_abort = true;
			}
		}

		float forwZ = -cosf(d_cameraParams.at(3))*cosf(d_cameraParams.at(4));
		float forwX = cosf(d_cameraParams.at(3))*sinf(d_cameraParams.at(4));
		float forwY = -sinf(d_cameraParams.at(3));

		float moveCoef = 0.1f;
		if (d_buttonPressed) {
			d_cameraParams.at(0) += forwX*moveCoef;
			d_cameraParams.at(1) += forwY*moveCoef;
			d_cameraParams.at(2) += forwZ*moveCoef;
		}


		// Writing the current position out
		if (cameraRail == -1) {
			static FILE *fileHandle = NULL;

			if (!fileHandle)
				fileHandle = fopen("camera.params", "w");

			fprintf(fileHandle, "%f %f %f %f %f\n", 
					d_cameraParams.at(0),
					d_cameraParams.at(1),
					d_cameraParams.at(2),
					d_cameraParams.at(3),
					d_cameraParams.at(4));
		}

		// Resetting cursor position
		XWarpPointer(d_dpy, d_win, d_win, 
				0, 0, d_winWidth, d_winHeight,
				zeroX, zeroY);
	} else {
		static FILE *fileHandle = NULL;

		if (!fileHandle)
			fileHandle = fopen("camera.params", "r");

		fscanf(fileHandle, "%f %f %f %f %f\n",
				&d_cameraParams.at(0),
				&d_cameraParams.at(1),
				&d_cameraParams.at(2),
				&d_cameraParams.at(3),
				&d_cameraParams.at(4));
	}

	if (!cameraRail && false)
		printf("Cameraparams: %f %f %f %f %f\n",
				d_cameraParams.at(0),
				d_cameraParams.at(1),
				d_cameraParams.at(2),
				d_cameraParams.at(3),
				d_cameraParams.at(4));

	setCamera(d_cameraParams.at(0),
			d_cameraParams.at(1),
			d_cameraParams.at(2),
			d_cameraParams.at(3),
			d_cameraParams.at(4));

}

void OpenGLController::hideCursor() {
	Cursor invisibleCursor;
	Pixmap bitmapNoData;
	XColor black;
	static char noData[] = { 0,0,0,0,0,0,0,0 };
	black.red = black.green = black.blue = 0;

	bitmapNoData = XCreateBitmapFromData(d_dpy, d_win, noData, 8, 8);
	invisibleCursor = XCreatePixmapCursor(d_dpy, bitmapNoData, bitmapNoData, 
			                                     &black, &black, 0, 0);
	XDefineCursor(d_dpy, d_win, invisibleCursor);
	XFreeCursor(d_dpy, invisibleCursor);
}

void OpenGLController::setCamera(float x, float y, float z, float rotX, float rotY) {
	if (!d_renderShaderData)
		throw std::string("Init rendering first");

	float width = 0.2f;
	float height = width*(float)d_depthHeight/(float)d_depthWidth;

	Matrix4 translate, rotateX, rotateY;
	d_projMatrix.perspective(-width, width, -height, height, width, 100.0f);
	translate.translate(-x, -y, -z);
	rotateX.rotate(rotX, 1.0f, 0.0f, 0.0f);
	rotateY.rotate(rotY, 0.0, 1.0f, 0.0f);

	// A bit of a hack but..
	/*Matrix4 invRotateX, invRotateY;
	invRotateX.rotate(-rotX, 1.0f, 0.0f, 0.0f);
	invRotateY.rotate(-rotY, 0.0, 1.0f, 0.0f);
	Matrix3 camRot(invRotateX*invRotateY);*/

	//Matrix4 final(translate*(rotateZ*(rotateY*proj)));
	Matrix4 modelProj(rotateX*(rotateY*translate));
	//Matrix4 modelProj(translate*(rotateX*rotateY));
	Matrix4 final(d_projMatrix*modelProj);
	/*Matrix4 normRot(final);
	normRot.invert();*/

	Matrix3 normRot(rotateX*rotateY);

	Shader *s = d_renderShaderData->shader();
	s->activate();
	glUniformMatrix4fv(glGetUniformLocation(s->handle(), "projMatrix"), 1, false, final.colMajorData());
	if (d_renderNormal)
		glUniformMatrix3fv(glGetUniformLocation(s->handle(), "normMatrix"), 1, false, normRot.colMajorData());

	//glUniform3f(glGetUniformLocation(s->handle(), "camPos"), x, y, z);
	//glUniformMatrix3fv(glGetUniformLocation(s->handle(), "camRotMatrix"), 1, false, camRot.colMajorData());

	checkErrors("set cam data");
}

Matrix4 OpenGLController::projMatrix() {
	return d_projMatrix;
}

void OpenGLController::loadMesh(Mesh &obj) {
	if (!d_renderShaderData)
		throw std::string("Init rendering before loading an object");

	d_meshPolys = obj.polyCount();
	int vertices = obj.vertCount();
	/*unsigned short shortIdx[d_meshPolys*3];
	for (int i = 0; i < d_mesPolys*3; ++i)
		shortIdx[i] = obj.indexData()[i];*/

	// Pushing the vertices in
	d_renderShaderData->addVData("vertPos", GL_FLOAT, 3, vertices, obj.vertData());
	if (d_renderNormal)
		d_renderShaderData->addVData("vertNorm", GL_FLOAT, 3, vertices, obj.normData());
	checkErrors("Pushing obj data");

	// And the indices
	glGenBuffers(1, &d_indexBuffer);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, d_indexBuffer);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(int)*d_meshPolys*3, obj.indexData(), GL_STATIC_DRAW);
	checkErrors("Creating index array");
}

// Gen normal:  0 = do not use normal, 1 = use separate normal, 2 = combine normal + depth
void OpenGLController::initRendering(int genNormal) {
	d_renderNormal = genNormal;

	// FBO
	//   Creating a texture for the depth component

	d_depthWidth = d_winWidth;
	d_depthHeight = d_winHeight;

	// Actually what we have here is first a depth buffer, and then a red buffer..
	// There are 2 reasons for this non-obvious choice of not using only a depth buffer:
	// (i) sharing a GL_DEPTH_COMPONENT32F buffer with CUDA does not work trivially
	// (ii) we have more freedom fiddling with the depth values in the fragment shader
	//      as the data is not actually used for Z culling.
	GLuint hfTex;
	d_depthAttachTex;
	glGenTextures(1, &hfTex);
	glGenTextures(1, &d_depthAttachTex);

	glBindTexture(GL_TEXTURE_2D, d_depthAttachTex);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT32F, d_depthWidth, d_depthHeight, 0, GL_DEPTH_COMPONENT, GL_FLOAT, NULL);

	if (genNormal < 2) {
		d_heightTex = new Texture(GL_TEXTURE_2D, hfTex, "depth buffer", d_depthWidth, d_depthHeight);
		d_heightTex->bind();
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_R32F, d_depthWidth, d_depthHeight, 0, GL_RED, GL_FLOAT, NULL);
		checkErrors("Creating FBO targets (hf/depth)");
	} else {
		d_heightTex = new Texture(GL_TEXTURE_2D, hfTex, "depth+normal buffer", d_depthWidth, d_depthHeight);
		d_heightTex->bind();
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32F, d_depthWidth, d_depthHeight, 0, GL_RGBA, GL_FLOAT, NULL);
		checkErrors("Creating FBO targets (hf+norm/depth)");
		d_normTex = NULL;
	}

	// Creating the FBO
	glGenFramebuffers(1, &d_renderFbo);
	glBindFramebuffer(GL_DRAW_FRAMEBUFFER, d_renderFbo);
	glFramebufferTexture2D(GL_DRAW_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_TEXTURE_2D, d_depthAttachTex, 0);
	glFramebufferTexture2D(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, hfTex, 0);
	checkErrors("FBO creation");

	// Creating the normal buffer as well
	if (genNormal == 1) {
		GLuint normTex;
		glGenTextures(1, &normTex);

		d_normTex = new Texture(GL_TEXTURE_2D, normTex, "normal texture", d_depthWidth, d_depthHeight);

		d_normTex->bind();
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA32F, d_depthWidth, d_depthHeight, 0, GL_RGBA, GL_FLOAT, NULL);
		glFramebufferTexture2D(GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT1, GL_TEXTURE_2D, normTex, 0);
		checkErrors("normal tex gen");
	}

	glBindFramebuffer(GL_DRAW_FRAMEBUFFER, 0);
	glBindTexture(GL_TEXTURE_2D, 0);

	// Blit sky borders
	Shader *s = g_defaultShaderPool->getShader("blitSky");
	s->activate();
	s->setVec2("borderFracs", d_borderFracW, d_borderFracH);

	// Shading
	s = g_defaultShaderPool->getShader(genNormal ? (genNormal == 2 ? "genDepthNormalUni" : "genDepthNormal") : "genDepth");
	s->activate();
	s->setVec2("winDims", (float)d_depthWidth, (float)d_depthHeight);
	d_renderShaderData = new ShaderData(s);

	// Let's start with identity matrix as the projection although it likely isn't going to be used
	Matrix4 identity;

	// FIXME:  You should have this in ShaderData..  But I'm lazy
	GLint projLoc = glGetUniformLocation(s->handle(), "projMatrix");
	if (projLoc == -1)
		throw std::string("Couldn't find projMatrix");
	glUniformMatrix4fv(projLoc, 1, false, identity.colMajorData());

	/*GLint camLoc = glGetUniformLocation(s->handle(), "camPos");
	if (camLoc == -1)
		throw std::string("Couldn't find camPos");
	glUniform3f(camLoc, 0.0f, 0.0f, 0.0f);*/

	if (genNormal) {
		GLint normLoc = glGetUniformLocation(s->handle(), "normMatrix");
		if (normLoc == -1)
			throw std::string("Couln't find normMatrix");
		glUniformMatrix3fv(normLoc, 1, false, Matrix3(identity).colMajorData());
	}

	// Creating the zero buffer
	glGenBuffers(1, &d_zeroBuffer);
	glBindBuffer(GL_PIXEL_PACK_BUFFER, d_zeroBuffer);

	size_t depthSize = sizeof(float)*d_depthWidth*d_depthHeight;

	if (genNormal == 2)
		depthSize *= 4;

	float *zeroData = (float*) malloc(depthSize);

	for (int i = 0; i < depthSize/sizeof(float); ++i)
		zeroData[i] = -30.0f;

	glBufferData(GL_PIXEL_PACK_BUFFER, depthSize, zeroData, GL_STATIC_READ);

	free(zeroData);
	glBindBuffer(GL_PIXEL_PACK_BUFFER, 0);

	checkErrors("set init render shaderdata");
}

void OpenGLController::blit(Texture *tex) {
	// I guess we could store some of these in a nicer way..
	glBindFramebuffer(GL_DRAW_FRAMEBUFFER, 0);
	glDisable(GL_DEPTH_TEST);
	glViewport(0, 0, d_winWidth, d_winHeight);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

	if (!d_blitTex) {
		d_blitTex = tex;
		if (g_defaultShaderPool) {
			Shader *s = g_defaultShaderPool->getShader("blitSky");
			d_blitShaderData = new ShaderData(s);
			d_blitShaderData->addTexture("srcTex", d_blitTex);
			Texture *t = new Texture(GL_TEXTURE_2D, d_depthAttachTex, "depth attachment");
			d_blitShaderData->addTexture("heightTex", t); //d_heightTex); 
			GLfloat texCoord[] = { 
				0.0f, 1.0f, 
				0.0f, 0.0f,
				1.0f, 1.0f,
				1.0f, 0.0f
			};
			d_blitShaderData->addVData("texCoord", GL_FLOAT, 2, 4, texCoord);
			// We can initially set pos the same 
			d_blitShaderData->addVData("pos", GL_FLOAT, 2, 4, texCoord);
		} else
			throw std::string("Couldn't find default shader pool");
	} else if (d_blitTex != tex)
		throw std::string("Changing of the blitted texture on the fly isn't yet allowed");

	d_blitShaderData->shader()->activate();
    d_blitShaderData->bind();
    // Drawing 4 primitives of triangle strip
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

	swapBuffers();
}

void OpenGLController::swapBuffers() {
	glXSwapBuffers(d_dpy, d_win);
    checkErrors("Swapping bufs");
}

int OpenGLController::getTexWidth(GLuint texRef) {
	return getTexParam(texRef, GL_TEXTURE_WIDTH);
}
int OpenGLController::getTexHeight(GLuint texRef) {
	return getTexParam(texRef, GL_TEXTURE_HEIGHT);
}
int OpenGLController::getTexChannels(GLuint texRef) {
	GLenum channels[] = {
		GL_TEXTURE_RED_SIZE,
		GL_TEXTURE_GREEN_SIZE,
		GL_TEXTURE_BLUE_SIZE,
		GL_TEXTURE_ALPHA_SIZE,
		GL_TEXTURE_LUMINANCE_SIZE,
		GL_TEXTURE_INTENSITY_SIZE,
		GL_TEXTURE_DEPTH_SIZE
	};

	int chanCount = 0;
	for (int i = 0; i < (int)(sizeof(channels)/sizeof(GLenum)); ++i)
		if (getTexParam(texRef, channels[i]))
			chanCount++;

	return chanCount;
}
int OpenGLController::getTexComponentSize(GLuint texRef) {
	GLenum channels[] = {
		GL_TEXTURE_RED_SIZE,
		GL_TEXTURE_GREEN_SIZE,
		GL_TEXTURE_BLUE_SIZE,
		GL_TEXTURE_ALPHA_SIZE,
		GL_TEXTURE_LUMINANCE_SIZE,
		GL_TEXTURE_INTENSITY_SIZE,
		GL_TEXTURE_DEPTH_SIZE
	};

	int chanMax = 0;
	for (int i = 0; i < (int)(sizeof(channels)/sizeof(GLenum)); ++i)
		if (getTexParam(texRef, channels[i]) > chanMax)
			chanMax = getTexParam(texRef, channels[i]);

	return chanMax;
}
