#include "opengl.h"
#include <cstdio>
#include <cstdlib>
#include <cmath>
#include <fstream>
#include <string.h>
#include <X11/Xatom.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>


using namespace SSEO;


WavefrontObject::WavefrontObject(std::string fname) : d_scale(1.0f), d_coordExchange(0) { //, float scale, bool addFloor) :
//	d_scale(scale)
//{
	/*d_addFloor = addFloor;
	d_minZ = 1e9f;*/
	loadFile(fname);
}

WavefrontObject::WavefrontObject() : d_scale(1.0f), d_coordExchange(0) {
}

void WavefrontObject::coordExchange(int c) {
	d_coordExchange = c;
}

void WavefrontObject::loadFile(std::string fname) {
	FILE *fd = fopen(fname.c_str(), "ro");
	if (!fd)
		throw std::string("Couldn't open object file ")+fname;

	char line[1024];
	line[1023] = 0;
	while (fgets(line, 1023, fd)) {
		//printf("read line %s\n", line);
		switch (line[0]) {
			case 'v':
				if (line[1] == ' ')
					pushVertex(line + 2);
				break;
			case 'f':
				pushTriangle(line + 2);
			/*case ' ':
			case '\n':
				continue;
			default:
				throw std::string("Garbage in the object file");*/
		}
	}

	printf("Loaded %ld points and %ld triangles from %s\n", d_vertices.size()/3, d_indices.size()/3, fname.c_str());
	split();
	printf("After split:  %ld points\n", d_vertices.size()/3);
	genNormals();
	printf("Generated normals\n");
}

void WavefrontObject::pushVertex(char *line) {
	float x, y, z;
	if (sscanf(line, "%f %f %f", &x, &y, &z) != 3)
		throw std::string("Couldn't parse a vertex from the object file");

	if (d_coordExchange == 1) {
		d_vertices.push_back(x*d_scale);
		d_vertices.push_back(z*d_scale);
		d_vertices.push_back(y*d_scale);
	} else {
		d_vertices.push_back(x*d_scale);
		d_vertices.push_back(y*d_scale);
		d_vertices.push_back(z*d_scale);
	}

	if (d_addFloor)
		d_minZ = fminf(d_vertices.at(d_vertices.size()-2), d_minZ);
}

void WavefrontObject::pushTriangle(char *line) {
	int index;
	int count = 0;
	std::vector<int> d_tempIndices;

	while (*line && sscanf(line, "%d", &index) == 1) {
		//printf("read index %d, linechar %c\n", index, *line);
		
		// If the index is negative, it denotes an index with the origo of the last vertex
		int newIndex = (index < 0) ? d_vertices.size()/3 + index : index - 1;
		d_tempIndices.push_back(newIndex);

		if (newIndex < 0 || newIndex >= d_vertices.size()/3)
			printf("Index %d (source %d) with %d vertices\n", newIndex, index, d_vertices.size()/3);

		count++;

		// Fast forwarding
		while (*line && *line != ' ')
			line++;
		// Now we are at the next whitespace..  Going on till the next non-white
		while (*line && *line == ' ')
			line++;
	}

	// Converting everything to triangles
	if (count < 3) {
		printf("Got a polygon with %d edges\n", count);
		throw std::string("We were prepared only for polygons");
	}

	for (int i = 2; i < (int)d_tempIndices.size(); ++i) {
		d_indices.push_back(d_tempIndices.at(0));
		d_indices.push_back(d_tempIndices.at(i-1));
		d_indices.push_back(d_tempIndices.at(i));
	}
}

void WavefrontObject::setScale(float s) {
	d_scale = s;
}

Mesh::Mesh() : d_minZ(1e9f), d_addFloor(false) {
}

size_t Mesh::polyCount() {
	return d_indices.size()/3;
}

size_t Mesh::vertCount() {
	return d_vertices.size()/3;
}

float *Mesh::vertData() {
	if (!d_vertices.size())
		return NULL;
	return &d_vertices.at(0);
}

void Mesh::addFloor(bool v) {
	d_addFloor = v;
}

void Mesh::split() {
	// If we are to add a floor, we introduce new vertices and new indices for it
	if (d_addFloor) {
		float planeExtent = 100.0f;
		int startVertex = d_vertices.size()/3;
		// v0
		d_vertices.push_back(-planeExtent);
		d_vertices.push_back(d_minZ);
		d_vertices.push_back(-planeExtent);
		// v1
		d_vertices.push_back(planeExtent);
		d_vertices.push_back(d_minZ);
		d_vertices.push_back(-planeExtent);
		// v2
		d_vertices.push_back(planeExtent);
		d_vertices.push_back(d_minZ);
		d_vertices.push_back(planeExtent);
		// v3
		d_vertices.push_back(-planeExtent);
		d_vertices.push_back(d_minZ);
		d_vertices.push_back(planeExtent);
		/*float3 v0(-planeExtent, -planeExtent, d_minZ);
		float3 v1(planeExtent, -planeExtent, d_minZ);
		float3 v2(planeExtent, planeExtent, d_minZ);
		float3 v3(-planeExtent, planeExtent, d_minZ);*/

		// First triangle
		d_indices.push_back(startVertex);
		d_indices.push_back(startVertex+1);
		d_indices.push_back(startVertex+2);

		// Second triangle
		d_indices.push_back(startVertex);
		d_indices.push_back(startVertex+2);
		d_indices.push_back(startVertex+3);
	}
	 
	std::vector<float> d_oldVertices = d_vertices;
	d_vertices.clear();

	int vertCounter = 0;

	if (d_indices.size()%3)
		throw std::string("Indices not divisible by 3");

	for (int poly = 0; poly < (int)d_indices.size()/3; ++poly) {
		int i0 = d_indices.at(poly*3 + 0);
		int i1 = d_indices.at(poly*3 + 1);
		int i2 = d_indices.at(poly*3 + 2);
		//printf("indices %d + 2, vertices %d\n", i0*3, d_oldVertices.size());
		float3 v0(d_oldVertices.at(i0*3 + 0), d_oldVertices.at(i0*3 + 1), d_oldVertices.at(i0*3 + 2));
		float3 v1(d_oldVertices.at(i1*3 + 0), d_oldVertices.at(i1*3 + 1), d_oldVertices.at(i1*3 + 2));
		float3 v2(d_oldVertices.at(i2*3 + 0), d_oldVertices.at(i2*3 + 1), d_oldVertices.at(i2*3 + 2));
		
		d_vertices.push_back(v0.x());
		d_vertices.push_back(v0.y());
		d_vertices.push_back(v0.z());
		d_indices.at(poly*3 + 0) = vertCounter;
		vertCounter++;
			
		d_vertices.push_back(v1.x());
		d_vertices.push_back(v1.y());
		d_vertices.push_back(v1.z());
		d_indices.at(poly*3 + 1) = vertCounter;
		vertCounter++;

		d_vertices.push_back(v2.x());
		d_vertices.push_back(v2.y());
		d_vertices.push_back(v2.z());
		d_indices.at(poly*3 + 2) = vertCounter;
		vertCounter++;
	}

	d_normals.clear(); // Normal data is invalidated
}

void Mesh::genNormals() {
	// Without extra hints, it's not really easy to generate normals the right way.
	// What we do here is that for each polygon, we calculate a normal, and push that normal
	// to each vertex.  Now there can be more than one polygon using a vertex, and in this
	// case averaging occurs.
	d_normals.clear();
	for (int i = 0; i < (int)d_vertices.size(); ++i)
		d_normals.push_back(0.0f);

	for (int poly = 0; poly < (int)d_indices.size()/3; ++poly) {
		int i0 = d_indices.at(poly*3 + 0);
		int i1 = d_indices.at(poly*3 + 1);
		int i2 = d_indices.at(poly*3 + 2);
		float3 v0(d_vertices.at(i0*3 + 0), d_vertices.at(i0*3 + 1), d_vertices.at(i0*3 + 2));
		float3 v1(d_vertices.at(i1*3 + 0), d_vertices.at(i1*3 + 1), d_vertices.at(i1*3 + 2));
		float3 v2(d_vertices.at(i2*3 + 0), d_vertices.at(i2*3 + 1), d_vertices.at(i2*3 + 2));

		float3 edge0 = v1 - v0;
		float3 edge1 = v2 - v1;

		float3 normal = edge0.cross(edge1);
		//float3 normal = edge1.cross(edge0);
		normal.normalize();

		// Scatter
		d_normals.at(i0*3 + 0) += normal.x();
		d_normals.at(i0*3 + 1) += normal.y();
		d_normals.at(i0*3 + 2) += normal.z();

		d_normals.at(i1*3 + 0) += normal.x();
		d_normals.at(i1*3 + 1) += normal.y();
		d_normals.at(i1*3 + 2) += normal.z();

		d_normals.at(i2*3 + 0) += normal.x();
		d_normals.at(i2*3 + 1) += normal.y();
		d_normals.at(i2*3 + 2) += normal.z();
	}

	// Let's normalize them once more
	for (int normI = 0; normI < (int)d_normals.size()/3; ++normI)
		float3::normalize(
				d_normals.at(normI*3 + 0),
				d_normals.at(normI*3 + 1),
				d_normals.at(normI*3 + 2));
}

float *Mesh::normData() {
	if (d_normals.size() != d_vertices.size())
		throw std::string("Normals not up to date");
	if (!d_normals.size())
		return NULL;
	return &d_normals.at(0);
}

int *Mesh::indexData() {
	if (!d_indices.size())
		return NULL;
	return &d_indices.at(0);
}


GLuint Texture::getHandle() {
	return d_id;
}

Texture::Texture(GLuint type, GLuint id, std::string name, int w, int h) :
    d_name(name), d_type(type), d_id(id), d_width(w), d_height(h) {
}

void Texture::fillFloats(std::string fname) {
	int fd = open(fname.c_str(), O_RDONLY);
	size_t size = sizeof(float)*4*width()*height();
	float *data = (float*) malloc(size);
	read(fd, data, size);
	
	fillFloats(data);

	free(data);
	close(fd);
}

void Texture::fillFloats(float *data) {
	bind();
	glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0,
			width(), height(), GL_RGBA, GL_FLOAT, data);
	OpenGLController::checkErrors("fill with float data");
}

Texture::~Texture() {
    glDeleteTextures(1, &d_id);
    std::string error = "glDeleteTextures";
	OpenGLController::checkErrors(error);
}

void Texture::bind() {
    if (d_id) {
        //glEnable(d_type);
        glBindTexture(d_type, d_id);
    } else {
        fprintf(stderr, "Can't bind an empty texture\n");
        exit(999);
    }

	OpenGLController::checkErrors("Binding texture \""+d_name+"\"");
}

int Texture::width() {
	// If it's not stored, we query for it
	if (d_width < 0)
		return OpenGLController::getTexWidth(d_id);
	else
	    return d_width;
}

int Texture::height() {
	if (d_height < 0)
		return OpenGLController::getTexHeight(d_id);
	else
	    return d_height;
}

int Texture::channels() {
	return OpenGLController::getTexChannels(d_id);
}

int Texture::channelSize() {
	return OpenGLController::getTexComponentSize(d_id);
}

void *Texture::getData(size_t *size) {
	#if 1
	//size_t actSize = (width()*height()*sizeof(float)*4);

	//printf("chans %d, chansize %d\n", channels(), channelSize());
	size_t actSize = (width()*height()*channels()*channelSize()/8);

	if (size)
		*size = actSize;

	float *data = (float*)malloc(actSize);
	for (int i = 0; i < (int)(actSize/4); ++i)
		data[i] = 0.0f;

	// In order to use get tex image, we need to have the texture bound
	bind();

	glGetTexImage(GL_TEXTURE_2D, 0, 
			(channels() == 1) ? GL_RED : GL_RGBA, 
			(channelSize() == 8) ? GL_UNSIGNED_BYTE : GL_FLOAT, 
			data);
	//glGetTexImage(GL_TEXTURE_2D, 0, GL_RGBA, GL_FLOAT, data);
	OpenGLController::checkErrors("Read out tex data");
	#else
	size_t actSize = (width()*height()*sizeof(unsigned int)*channels()); //channels()*channelSize())/8;

	if (size)
		*size = actSize;

	unsigned char *data = (unsigned char*)malloc(actSize);
	for (int i = 0; i < actSize; ++i)
		data[i] = 0;

	// In order to use get tex image, we need to have the texture bound
	bind();

	glGetTexImage(GL_TEXTURE_2D, 0, 
			(channels() == 1) ? GL_R : GL_RGBA,
			GL_UNSIGNED_BYTE, data);
	OpenGLController::checkErrors("Read out tex data");

	#endif

	return data;
}

void Texture::writeToFile(std::string fname) {
	FILE *out = fopen(fname.c_str(), "w+");
	size_t size;
	float *data = (float*)getData(&size);
	size_t written = fwrite(data, sizeof(float), size/sizeof(float), out);
	printf("Wrote %ld bytes\n", written*sizeof(float));
	fclose(out);
}

std::string Texture::name() {
    return d_name;
}


Shader::Shader(std::string name, std::string src, std::vector<std::string> outs) : d_name(name) {
    d_programHandle = glCreateProgram();
    GLuint vp = glCreateShader(GL_VERTEX_SHADER);
    GLuint fp = glCreateShader(GL_FRAGMENT_SHADER);

    std::string vp_src, fp_src;
    try {
        vp_src = readSrc(src + ".vp");
        fp_src = readSrc(src + ".fp");
    } catch (std::string e) {
        fprintf(stderr, "Couldn't read sources for shader %s: %s\n",
                d_name.c_str(), e.c_str());
        exit(199);
    }
    const char *vp_src_ptr = vp_src.c_str();
    const char *fp_src_ptr = fp_src.c_str();

	const char *vpSrc[] = {
		"#version 420\n",
		vp_src.c_str()
	};

	const char *fpSrc[] = {
		"#version 420\n",
		fp_src.c_str()
	};

    //glShaderSource(vp, 1, &vp_src_ptr, NULL);
    glShaderSource(vp, 2, vpSrc, NULL);

    glCompileShader(vp);
    int rvalue;
    glGetShaderiv(vp, GL_COMPILE_STATUS, &rvalue);
    if (!rvalue) {
        printf("Error in compiling vp\n");
        GLchar log[10240];
        GLsizei length;
        glGetShaderInfoLog(vp, 10239, &length, log);
        printf("VP log: %s\n", log);
        exit(200);
    }
    glAttachShader(d_programHandle, vp);

    //glShaderSource(fp, 1, &fp_src_ptr, NULL);
    glShaderSource(fp, 2, fpSrc, NULL);

    glCompileShader(fp);
    glGetShaderiv(fp, GL_COMPILE_STATUS, &rvalue);
    if (!rvalue) {
        printf("Error in compiling fp\n");
        GLchar log[10240];
        GLsizei length;
        glGetShaderInfoLog(fp, 10239, &length, log);
        printf("FP log:\n%s\n", log);
        exit(200);
    }
    glAttachShader(d_programHandle, fp);

	// Setting the outputs
	for (int i = 0; i < outs.size(); ++i)
		glBindFragDataLocation(d_programHandle, i, outs.at(i).c_str());
    OpenGLController::checkErrors("Data locations");

    glLinkProgram(d_programHandle);
    
    glGetProgramiv(d_programHandle, GL_LINK_STATUS, &rvalue);
    if (!rvalue) {
        printf("Error in linking sp\n");
        GLchar log[10240];
        GLsizei length;
        glGetProgramInfoLog(d_programHandle, 10239, &length, log);
        printf("SP log:\n%s\n", log);
        exit(201);
    }   
    
    OpenGLController::checkErrors("Shader compilation");
}   
    
std::string Shader::name() {
    return d_name;
}   
    
std::string Shader::readSrc(std::string fileName) {
    std::ifstream fin;
    fin.open(fileName.c_str(), std::ios::in);

    char linebuffer[1024];
    std::string filestring; // Whole file in one string.
    
    if (!fin)
        throw "Couldn't open file " + fileName;

    while (!fin.eof()) {
        fin.getline(linebuffer, 1023);
        // For each line, we delete //-comments from the end.
        std::string line(linebuffer);
        // The following line is a crop from the beginning to "//" (or to the end if not existing). 
        std::string cropline(line, 0, line.find("//"));
        filestring += cropline;
    }

    return filestring;
}

void Shader::setVec2(std::string name, float a, float b) {
	GLint loc = glGetUniformLocation(d_programHandle, name.c_str());
	glUniform2f(loc, a, b);

    OpenGLController::checkErrors("Trying to set vec2 uniform for shader");
}

ShaderData::ShaderData(Shader* s) : d_shader(s) {
    glGenVertexArrays(1, &d_vao);
    OpenGLController::checkErrors(std::string("Creating Vertex Object Array for shader ") + d_shader->name());
}

ShaderData::~ShaderData() {
    glDeleteVertexArrays(1, &d_vao);
    std::string error = std::string("Removing VAO for shader ") + d_shader->name();
    OpenGLController::checkErrors(error);

    // Textures can stay, but we're removing buffer objects
    int counter = 0;
    for (std::list<std::pair<std::pair<std::string, size_t>, GLuint> >::iterator i = d_vbufs.begin();
            i != d_vbufs.end();
            ++i, ++counter) {
        glDeleteBuffers(1, &(*i).second);
        error = "For shader " + d_shader->name();
        OpenGLController::checkErrors(error);
    }
}

void ShaderData::addTexture(std::string varName, Texture *tex) {
    // During binding, we're finding the address of the variable and assigning
    // it to a texture unit where tex gets loaded.
    d_textures.push_back(std::pair<std::string, Texture*>(varName, tex));
}

void ShaderData::addVData(std::string varName, GLint type, int width, size_t length, void *data) {
    // Data is added differently than the textures.
    // When new data comes, we restore the vertex array object
    // and push new data in.
    glBindVertexArray(d_vao);

    GLuint vbuf;
    GLint varPtr;

    // Creating the buffer object
    glGenBuffers(1, &vbuf);
	OpenGLController::checkErrors("Creating VBO for shader " + d_shader->name());

    // Just figuring out how much data there actually is
    int elemSize;
    if (type == GL_FLOAT || type == GL_INT || type == GL_UNSIGNED_INT)
        elemSize = 4;
    else if (type == GL_BYTE || type == GL_UNSIGNED_BYTE)
        elemSize = 1;
    else
        throw std::string("Got unknown vertex data type for shader ") + d_shader->name();

    // Pushing in data for the buffer
    pushVData(vbuf, length*width*elemSize, data);

    // Binding it to the variable in the shader
    varPtr = glGetAttribLocation(d_shader->d_programHandle, varName.c_str());
	OpenGLController::checkErrors("Unable to find variable " + varName + " in shader " + d_shader->name());
    glVertexAttribPointer(varPtr, width, type, GL_FALSE, 0, NULL);
    OpenGLController::checkErrors("Unable to bind buffer object to variable " + varName + " in shader " + d_shader->name());
    glEnableVertexAttribArray(varPtr);

    glBindVertexArray(0);

    d_vbufs.push_back(std::pair<std::pair<std::string, size_t>, GLuint>(std::pair<std::string, size_t>(varName, length*width*elemSize), vbuf));
}

void ShaderData::changeVData(std::string varName, void *data) {
    // Finding the element with the correct variable name
    for (std::list<std::pair<std::pair<std::string, size_t>, GLuint> >::iterator i = d_vbufs.begin();
            i != d_vbufs.end(); ++i)
        if ((*i).first.first == varName) {
            pushVData((*i).second, (*i).first.second, data);
            glBindVertexArray(0);
            return;
        }
    fprintf(stderr, "Tried to change VData %s for shader %s, but hasn't been added\n", varName.c_str(), d_shader->name().c_str());
}

// It's important to note that this function leaves the buffer bound
void ShaderData::pushVData(GLuint bufId, size_t size, void *data) {
    glBindBuffer(GL_ARRAY_BUFFER, bufId);
    glBufferData(GL_ARRAY_BUFFER, size, data, GL_STREAM_DRAW);
    OpenGLController::checkErrors("Pushing VBO data in for shader " + d_shader->name());
}

void Shader::activate() {
    glUseProgram(d_programHandle);
    OpenGLController::checkErrors(std::string("Using program on shader ") + name());
}

void ShaderData::bind() {
    //printf("d_vao: %d\n", d_vao);
    OpenGLController::checkErrors(std::string("Before binding VAO"));
    glBindVertexArray(d_vao);
    OpenGLController::checkErrors(std::string("Binding VAO"));
    // Going through textures one by one, and assigning texturing units
    // consecutively.
    GLuint texCounter = 0;
    for (std::list<std::pair<std::string, Texture*> >::iterator i = d_textures.begin();
        i != d_textures.end();
        i++) {
        // Binding counter to the name
        /*printf("Binding to handle %d, variable \"%s\", texCounter %d\n",
                shader()->handle(), (*i).first.c_str(), texCounter);*/
        glUniform1i(glGetUniformLocation(shader()->handle(), (*i).first.c_str()), texCounter);
        OpenGLController::checkErrors(std::string("Binding texture variable ") + (*i).first + ". ");
        // Binding texture to the corresponding unit
        glActiveTexture(GL_TEXTURE0 + texCounter);
        (*i).second->bind();
        OpenGLController::checkErrors(std::string("Binding texture ") + (*i).first + ". ");
        texCounter++;
    }
}

Shader *ShaderData::shader() {
    return d_shader;
}

GLuint Shader::handle() {
    return d_programHandle;
}

ShaderPool::ShaderPool() {
}

Shader* ShaderPool::getShader(std::string name) {
    for (std::list<Shader*>::iterator i = d_shaders.begin(); i != d_shaders.end(); ++i)
        if ((*i)->name() == name)
            return *i;

    // No match, we have to load it
	// FIXME:  This is not a right way to pass output params
	std::vector<std::string> outputs;

	if (name == std::string("blit") || name == std::string("blitSky")) {
		outputs.push_back("color");
	} else if (name == std::string("genDepthNormalUni")) {
		outputs.push_back("screenDepthNormal");
	} else if (name == std::string("genDepthNormal")) {
		outputs.push_back("screenDepth");
		outputs.push_back("screenNormal");
	} else if (name == std::string("hbao") || name == std::string("mipOcc")) {
		outputs.push_back("occlusionOut");
	} else
		throw std::string("No outputs for shader ") + name;
		
    Shader *sh = new Shader(name, std::string("shaders/")+name, outputs);

    d_shaders.push_back(sh);
    return sh;
}
